<?php
// ===================
// Kalenderblatt 2025  (Letzte Überarbeitung: 4. Januar 2026)
// Dieses Script und alle zugehörigen Dateien im Ordner "kalenderdaten" unterliegen dem Urheberrecht S.G.Schiffer 2025 - sgs-visual.de
// Die Dateien dürfen weder zum Verkauf noch zum Herunterladen von anderen Webseiten als sgs-visual.de angeboten werden.
// Sie erhalten ein kostenloses Nutzungsrecht, was aber nicht bedeutet, das die Dateien damit in Ihr Eigentum übergehen.
// Die Copyright-Informationen dürfen nicht verändert oder gelöscht werden.
// ===================

// Bundesland für Feiertagsanzeige (bitte Ihr eigenes Bundesland aus folgende Zeile kopieren und in Zeile 13, wie gezeigt, einfügen):
// Baden-Württemberg, Bayern, Berlin, Brandenburg, Bremen, Hamburg, Hessen, Mecklenburg-Vorpommern,
// Niedersachsen, Nordrhein-Westfalen, Rheinland-Pfalz, Saarland, Sachsen, Sachsen-Anhalt, Schleswig-Holstein, Thüringen, Augsburg

$bundesland = "Hessen"; // HIER Ihr Bundesland einfügen (z.B.:  $bundesland = "Hessen"; )

// Koordinaten für Sonnenauf-/untergang (Wiesbaden) bitte an Ihren Wohnort anpassen
$breitengrad = 50.0463333;
$laengengrad = 8.1976777;

//   ######   Ab hier keine Änderungen mehr vornehmen !   ######

// --- MONDPHASEN-BERECHNUNG ---
define('EPOCH', 2444238.5);
define('ELONGE', 278.833540);
define('ELONGP', 282.596403);
define('ECCENT', 0.016718);
define('MMLONG', 64.975464);
define('MMLONGP', 349.383063);
define('MLNODE', 151.950429);
define('MINC', 5.145396);
define('MECC', 0.054900);
define('MSMAX', 384401.0);
define('SYNMONTH', 29.53058868);

function fixangle($a) { return $a - 360.0 * floor($a / 360.0); }
function torad($a) { return $a * (pi() / 180.0); }
function kepler($m, $ecc) {
    $EPSILON = 1e-6;
    $m = torad($m);
    $e = $m;
    do {
        $delta = $e - $ecc * sin($e) - $m;
        $e -= $delta / (1 - $ecc * cos($e));
    } while (abs($delta) > $EPSILON);
    return $e;
}
function jtime($timestamp) {
    return ($timestamp / 86400) + 2440587.5;
}
function getMoonPhaseString($timestamp = null) {
    if ($timestamp === null) $timestamp = time();
    $pdate = jtime($timestamp);

    // Sonne
    $Day = $pdate - EPOCH;
    $N = fixangle((360 / 365.2422) * $Day);
    $M = fixangle($N + ELONGE - ELONGP);
    $Ec = kepler($M, ECCENT);
    $Ec = sqrt((1 + ECCENT) / (1 - ECCENT)) * tan($Ec / 2);
    $Ec = 2 * (180.0 / pi()) * atan($Ec);
    $Lambdasun = fixangle($Ec + ELONGP);

    // Mond
    $ml = fixangle(13.1763966 * $Day + MMLONG);
    $MM = fixangle($ml - 0.1114041 * $Day - MMLONGP);
    $MN = fixangle(MLNODE - 0.0529539 * $Day);
    $Ev = 1.2739 * sin(torad(2 * ($ml - $Lambdasun) - $MM));
    $Ae = 0.1858 * sin(torad($M));
    $A3 = 0.37 * sin(torad($M));
    $MmP = $MM + $Ev - $Ae - $A3;
    $mEc = 6.2886 * sin(torad($MmP));
    $A4 = 0.214 * sin(torad(2 * $MmP));
    $lP = $ml + $Ev + $mEc - $Ae + $A4;
    $V = 0.6583 * sin(torad(2 * ($lP - $Lambdasun)));
    $lPP = $lP + $V;

    // Mondalter in Grad
    $MoonAge = fixangle($lPP - $Lambdasun);

    // Mondphase als Prozent (ohne Nachkommastellen)
    $MoonPhase = (1 - cos(torad($MoonAge))) / 2;
    $percent = round($MoonPhase * 100);

    if ($percent < 1) {
        $out = "Neumond {$percent}%";
    } elseif ($percent > 99) {
        $out = "Vollmond {$percent}%";
    } elseif ($MoonAge < 180) {
        $out = "zun. Mond {$percent}%";
    } else {
        $out = "abn. Mond {$percent}%";
    }
    return $out;
}
// --- ENDE MONDPHASEN-BERECHNUNG ---

//   ---  Daten aus der Feiertags-JSON-Datei laden, abhängig von Ihrem oben eingegebenen Bundesland  ---

$feiertage_json = file_get_contents(__DIR__ . '/kalenderdaten/feiertage.json');
$feiertage_data = json_decode($feiertage_json, true);

//  --- Weitere Formatierungen und Funktionalitäten ---

$now = new DateTime('now', new DateTimeZone('Europe/Berlin'));
$heute = $now->format('d.m.Y');
$heute_kurz = $now->format('d.m');
$jahr = $now->format('Y');
$monat = $now->format('n');
$tag = (int)$now->format('j');
$kalenderwoche = (int)$now->format('W');
$tag_im_jahr = (int)$now->format('z') + 1;

function getMoonPhaseStringFull($timestamp) {
    $phase = getMoonPhaseString($timestamp);
    $phaseLower = strtolower($phase);
    if (strpos($phaseLower, 'vollmond') !== false || $phaseLower === 'voll') {
        return 'Vollmond';
    }
    if (strpos($phaseLower, 'neumond') !== false || $phaseLower === 'neu') {
        return 'Neumond';
    }
    return $phase;
}
$mondphase = getMoonPhaseStringFull($now->getTimestamp());

$sun_info = date_sun_info($now->getTimestamp(), $breitengrad, $laengengrad);
$sonnenaufgang = date('H:i', $sun_info['sunrise']);
$sonnenuntergang = date('H:i', $sun_info['sunset']);

function getJahreszeit($now) {
    $jahr = $now->format('Y');
    $zeiten = [
        ['von'=>"$jahr-01-01", 'bis'=>"$jahr-03-19", 'text'=>"Winter 21.12. - 19.03."],
        ['von'=>"$jahr-03-20", 'bis'=>"$jahr-06-20", 'text'=>"Frühling 20.03. - 20.06."],
        ['von'=>"$jahr-06-21", 'bis'=>"$jahr-09-22", 'text'=>"Sommer 21.06. - 22.09."],
        ['von'=>"$jahr-09-23", 'bis'=>"$jahr-12-20", 'text'=>"Herbst 23.09. - 20.12."],
        ['von'=>"$jahr-12-21", 'bis'=>"$jahr-12-31", 'text'=>"Winter 21.12. - 19.03."],
    ];
    $timestamp = $now->getTimestamp();
    foreach ($zeiten as $z) {
        $von = strtotime($z['von']);
        $bis = strtotime($z['bis']) + 86399;
        if ($timestamp >= $von && $timestamp <= $bis) {
            return $z['text'];
        }
    }
    return '';
}
$jahreszeit = getJahreszeit($now);

$heute_feiertag = '';
$heute_sonstiger = '';
$feiertage = [];
$sonstige_tage = [];
if (isset($feiertage_data['years'][$jahr])) {
    $feiertage = $feiertage_data['years'][$jahr]['holidays'];
    $sonstige_tage = $feiertage_data['years'][$jahr]['other_days'];
}
if (!empty($feiertage)) {
    foreach ($feiertage as $h) {
        if ($h['date'] === $heute) {
            if (in_array('all', $h['states']) || in_array($bundesland, $h['states'])) {
                $heute_feiertag = $h['name'];
                break;
            }
        }
    }
}
if (!$heute_feiertag && !empty($sonstige_tage)) {
    foreach ($sonstige_tage as $o) {
        if ($o['date'] === $heute) {
            $heute_sonstiger = $o['name'];
            break;
        }
    }
}

$monate = [
    1=>"Januar",2=>"Februar",3=>"März",4=>"April",5=>"Mai",6=>"Juni",
    7=>"Juli",8=>"August",9=>"September",10=>"Oktober",11=>"November",12=>"Dezember"
];
$monat_name = $monate[$monat];

function getSternzeichen($now) {
    $jahr = $now->format('Y');
    $sternzeichen = [
        ['von'=>"$jahr-01-01", 'bis'=>"$jahr-01-20", 'name'=>"Steinbock", 'zeit'=>"22.12. - 20.01."],
        ['von'=>"$jahr-01-21", 'bis'=>"$jahr-02-19", 'name'=>"Wassermann", 'zeit'=>"21.01. - 19.02."],
        ['von'=>"$jahr-02-20", 'bis'=>"$jahr-03-20", 'name'=>"Fische", 'zeit'=>"20.02. - 20.03."],
        ['von'=>"$jahr-03-21", 'bis'=>"$jahr-04-20", 'name'=>"Widder", 'zeit'=>"21.03. - 20.04."],
        ['von'=>"$jahr-04-21", 'bis'=>"$jahr-05-20", 'name'=>"Stier", 'zeit'=>"21.04. - 20.05."],
        ['von'=>"$jahr-05-21", 'bis'=>"$jahr-06-21", 'name'=>"Zwilling", 'zeit'=>"21.05. - 21.06."],
        ['von'=>"$jahr-06-22", 'bis'=>"$jahr-07-22", 'name'=>"Krebs", 'zeit'=>"22.06. - 22.07."],
        ['von'=>"$jahr-07-23", 'bis'=>"$jahr-08-23", 'name'=>"Löwe", 'zeit'=>"23.07. - 23.08."],
        ['von'=>"$jahr-08-24", 'bis'=>"$jahr-09-23", 'name'=>"Jungfrau", 'zeit'=>"24.08. - 23.09."],
        ['von'=>"$jahr-09-24", 'bis'=>"$jahr-10-23", 'name'=>"Waage", 'zeit'=>"24.09. - 23.10."],
        ['von'=>"$jahr-10-24", 'bis'=>"$jahr-11-22", 'name'=>"Skorpion", 'zeit'=>"24.10. - 22.11."],
        ['von'=>"$jahr-11-23", 'bis'=>"$jahr-12-21", 'name'=>"Schütze", 'zeit'=>"23.11. - 21.12."],
        ['von'=>"$jahr-12-22", 'bis'=>"$jahr-12-31", 'name'=>"Steinbock", 'zeit'=>"22.12. - 20.01."],
    ];
    $timestamp = $now->getTimestamp();
    foreach ($sternzeichen as $sz) {
        $von = strtotime($sz['von']);
        $bis = strtotime($sz['bis']) + 86399;
        if ($timestamp >= $von && $timestamp <= $bis) {
            return "<b>{$sz['name']}</b> <span style='font-weight:normal;'>({$sz['zeit']})</span>";
        }
    }
    return '';
}
$sternzeichen = getSternzeichen($now);

setlocale(LC_TIME, 'de_DE.UTF-8');
$wochentag_en = $now->format('l');
$wochentage = [
    'Monday'=>'Montag','Tuesday'=>'Dienstag','Wednesday'=>'Mittwoch','Thursday'=>'Donnerstag',
    'Friday'=>'Freitag','Saturday'=>'Samstag','Sunday'=>'Sonntag'
];
$wochentag = $wochentage[$wochentag_en] ?? $wochentag_en;

$tag_farbe = "#000";
if ($heute_feiertag || $wochentag_en === 'Sunday') {
    $tag_farbe = "#FF0000";
}

$feiertag_ausgabe = "";
if ($heute_feiertag) {
    $feiertag_ausgabe = "<span style='color:#FF0000;font-weight:bold;font-size:12px;'>$heute_feiertag</span>";
} elseif ($heute_sonstiger) {
    $feiertag_ausgabe = "<span style='color:#EB7D29;font-weight:600;font-size:12px;'>$heute_sonstiger</span>";
} else {
    $feiertag_ausgabe = "<span style='font-size:12px;'>&nbsp;</span>";
}

function getMoonPhaseType($timestamp) {
    $phaseStr = strtolower(getMoonPhaseString($timestamp));
    if (strpos($phaseStr, 'neumond') !== false || strpos($phaseStr, 'neu') !== false) {
        return 'new';
    } elseif (strpos($phaseStr, 'vollmond') !== false || strpos($phaseStr, 'voll') !== false) {
        return 'full';
    } elseif (strpos($phaseStr, 'zun') !== false) {
        return 'waxing';
    } elseif (strpos($phaseStr, 'abn') !== false) {
        return 'waning';
    }
    return 'new';
}
$mondphase_type = getMoonPhaseType($now->getTimestamp());

function getMoonPhasePercent($timestamp) {
    $phaseStr = getMoonPhaseString($timestamp);
    if (preg_match('/(\d+)\s*%/', $phaseStr, $m)) {
        return (int)$m[1];
    }
    return 0;
}
$mondphase_percent = getMoonPhasePercent($now->getTimestamp());

function getMoonPhaseEmoji($percent, $type) {
    if ($percent <= 5) return '🌑'; // Neumond
    if ($percent >= 95) return '🌕'; // Vollmond
    if ($type === 'waxing') { // zunehmend
        if ($percent < 38) return '🌒';
        if ($percent < 63) return '🌓';
        return '🌔';
    }
    if ($type === 'waning') { // abnehmend
        if ($percent > 62) return '🌖';
        if ($percent > 37) return '🌗';
        return '🌘';
    }
    return '🌑'; // Fallback
}
$mond_emoji = getMoonPhaseEmoji($mondphase_percent, $mondphase_type);

$mond_text = '';
if ($mondphase_percent == 0) {
    $mond_text = 'Neumond';
} elseif ($mondphase_percent == 100) {
    $mond_text = 'Vollmond';
} elseif ($mondphase_type === 'waxing') {
    $mond_text = 'zun. Mond';
} elseif ($mondphase_type === 'waning') {
    $mond_text = 'abn. Mond';
}

/* --- BEGINN: Nur die nächste zukünftige Mondphase anzeigen --- */
function naechsteMondphaseMitText() {
    define('SYNMONTH2', 29.53058868);
    function torad2($arg) { return ($arg * (pi() / 180.0)); }
    function dsin2($arg) { return (sin(torad2($arg))); }
    function dcos2($arg) { return (cos(torad2($arg))); }
    function jtime2($timestamp) { return ($timestamp / 86400) + 2440587.5; }
    function jdaytosecs2($jday=0) { return ($jday - 2440587.5) * 86400; }
    function jyear2($td, &$yy, &$mm, &$dd) {
        $td += 0.5;
        $z = floor($td);
        $f = $td - $z;
        if ($z < 2299161.0) {
            $a = $z;
        } else {
            $alpha = floor(($z - 1867216.25) / 36524.25);
            $a = $z + 1 + $alpha - floor($alpha / 4);
        }
        $b = $a + 1524;
        $c = floor(($b - 122.1) / 365.25);
        $d = floor(365.25 * $c);
        $e = floor(($b - $d) / 30.6001);
        $dd = $b - $d - floor(30.6001 * $e) + $f;
        $mm = $e < 14 ? $e - 1 : $e - 13;
        $yy = $mm > 2 ? $c - 4716 : $c - 4715;
    }
    function meanphase2($sdate, $k) {
        $t = ($sdate - 2415020.0) / 36525;
        $t2 = $t * $t;
        $t3 = $t2 * $t;
        return 2415020.75933 + SYNMONTH2 * $k
            + 0.0001178 * $t2
            - 0.000000155 * $t3
            + 0.00033 * dsin2(166.56 + 132.87 * $t - 0.009173 * $t2);
    }
    function truephase2($k, $phase) {
        $apcor = 0;
        $k += $phase;
        $t = $k / 1236.85;
        $t2 = $t * $t;
        $t3 = $t2 * $t;
        $pt = 2415020.75933
            + SYNMONTH2 * $k
            + 0.0001178 * $t2
            - 0.000000155 * $t3
            + 0.00033 * dsin2(166.56 + 132.87 * $t - 0.009173 * $t2);
        $m = 359.2242 + 29.10535608 * $k - 0.0000333 * $t2 - 0.00000347 * $t3;
        $mprime = 306.0253 + 385.81691806 * $k + 0.0107306 * $t2 + 0.00001236 * $t3;
        $f = 21.2964 + 390.67050646 * $k - 0.0016528 * $t2 - 0.00000239 * $t3;
        if (($phase < 0.01) || (abs($phase - 0.5) < 0.01)) {
            $pt += (0.1734 - 0.000393 * $t) * dsin2($m)
                + 0.0021 * dsin2(2 * $m)
                - 0.4068 * dsin2($mprime)
                + 0.0161 * dsin2(2 * $mprime)
                - 0.0004 * dsin2(3 * $mprime)
                + 0.0104 * dsin2(2 * $f)
                - 0.0051 * dsin2($m + $mprime)
                - 0.0074 * dsin2($m - $mprime)
                + 0.0004 * dsin2(2 * $f + $m)
                - 0.0004 * dsin2(2 * $f - $m)
                - 0.0006 * dsin2(2 * $f + $mprime)
                + 0.0010 * dsin2(2 * $f - $mprime)
                + 0.0005 * dsin2($m + 2 * $mprime);
            $apcor = 1;
        } elseif ((abs($phase - 0.25) < 0.01 || (abs($phase - 0.75) < 0.01))) {
            $pt += (0.1721 - 0.0004 * $t) * dsin2($m)
                + 0.0021 * dsin2(2 * $m)
                - 0.6280 * dsin2($mprime)
                + 0.0089 * dsin2(2 * $mprime)
                - 0.0004 * dsin2(3 * $mprime)
                + 0.0079 * dsin2(2 * $f)
                - 0.0119 * dsin2($m + $mprime)
                - 0.0047 * dsin2($m - $mprime)
                + 0.0003 * dsin2(2 * $f + $m)
                - 0.0004 * dsin2(2 * $f - $m)
                - 0.0006 * dsin2(2 * $f + $mprime)
                + 0.0021 * dsin2(2 * $f - $mprime)
                + 0.0003 * dsin2($m + 2 * $mprime)
                + 0.0004 * dsin2($m - 2 * $mprime)
                - 0.0003 * dsin2(2 * $m + $mprime);
            if ($phase < 0.5) {
                $pt += 0.0028 - 0.0004 * dcos2($m) + 0.0003 * dcos2($mprime);
            } else {
                $pt += -0.0028 + 0.0004 * dcos2($m) - 0.0003 * dcos2($mprime);
            }
            $apcor = 1;
        }
        if (!$apcor) {
            return 0; // <--- WICHTIG: statt exit, return 0 verwenden!
        }
        return $pt;
    }
    function phasehunt2($time = -1) {
        if (empty($time) || $time == -1) {
            $time = time();
        }
        $sdate = jtime2($time);
        $adate = $sdate - 45;
        jyear2($adate, $yy, $mm, $dd);
        $k1 = floor(($yy + (($mm - 1) * (1.0 / 12.0)) - 1900) * 12.3685);
        $adate = $nt1 = meanphase2($adate, $k1);
        while (1) {
            $adate += SYNMONTH2;
            $k2 = $k1 + 1;
            $nt2 = meanphase2($adate, $k2);
            if (($nt1 <= $sdate) && ($nt2 > $sdate)) {
                break;
            }
            $nt1 = $nt2;
            $k1 = $k2;
        }
        return array(
            jdaytosecs2(truephase2($k1, 0.0)),   // 0: vorheriger Neumond
            jdaytosecs2(truephase2($k1, 0.25)),  // 1: erstes Viertel
            jdaytosecs2(truephase2($k1, 0.5)),   // 2: Vollmond
            jdaytosecs2(truephase2($k1, 0.75)),  // 3: letztes Viertel
            jdaytosecs2(truephase2($k2, 0.0))    // 4: nächster Neumond
        );
    }

    $now = time();
    $phases = phasehunt2($now);

    $vollmond = $phases[2];
    $neumond = $phases[4];

    if ($vollmond > $now && ($vollmond < $neumond || $neumond <= $now)) {
        return "Nächster Vollmond: " . date('d.m.Y', $vollmond);
    }
    if ($neumond > $now) {
        return "Nächster Neumond: " . date('d.m.Y', $neumond);
    }
    return "";
}
$mondphase_next_text = naechsteMondphaseMitText();
/* --- ENDE: zukünftige Mondphase anzeigen --- */
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="utf-8">
    <title>Kalenderblatt</title>
    <meta name="author" content="S.G.Schiffer">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
    .kalender-canvas {
        width: 200px;
        max-width: 98vw;
        height: 270px;
        background: #fff;
        border: 1px solid #555;
        border-radius: 2px;
        box-shadow: 0 2px 12px #bbb;
        margin: 0 auto;
        font-family: Arial, sans-serif;
        display: flex;
        flex-direction: column;
        overflow: hidden;
    }
    .kalender-header {
        background: linear-gradient(90deg, #2C79DC 0%, #4F9DFE 100%);
        color: #fff;
        font-family: Tahoma, sans-serif;
        font-size: 1.3rem;
        font-weight: 600;
        text-align: center;
        padding: 12px 0 8px 0;
        letter-spacing: 1px;
    }
    .kalender-body {
        background: #fff;
        flex: 1 1 auto;
        display: flex;
        flex-direction: column;
        align-items: center;
        padding: 0 0 0 0;
    }
    .kalender-zeile {
        width: 100%;
        text-align: center;
        margin: 0;
        padding: 0;
    }
    .kalender-info {
        font-size: 8px;
        color: #222;
        margin-top: 6px;
        margin-bottom: 0;
        letter-spacing: 0.5px;
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 6px;
    }
    .kalender-mondphase-next {
        font-size: 8px;
        color: #000;
        margin: 0 0 2px 0;
        text-align: center;
        display: block;
        width: 100%;
    }
    .kalender-sonne {
        font-size: 8px;
        color: #000;
        margin: 2px 0 0 0;
    }
    .kalender-jahreszeit {
        font-size: 8px;
        color: #000;
        margin: 2px 0 0 0;
    }
    .kalender-tag {
        font-size: 86px;
        font-weight: 600;
        margin: 8px 0 0 0;
        color: <?= $tag_farbe ?>;
        line-height: 80%;
    }
    .kalender-feiertag {
        margin: 2px 0 0 0;
        min-height: 18px;
    }
    .kalender-monatjahr {
        font-size: 20px;
        color: #222;
        margin: 2px 0 0 0;
        font-weight: 700;
    }
    .kalender-sternzeichen {
        font-size: 8px;
        color: #000;
        margin: 2px 0 0 0;
    }
    .kalender-footer {
        font-size: 8px;
        color: #000;
        margin-top: 10px;
        margin-left: 40px;
        margin-bottom: 0;
        display: flex;
        justify-content: space-between;
        align-items: center;
        width: 96%;
    }
    .kalender-footer a {
        color: #FF0000;
        text-decoration: none;
        font-weight: bold;
        font-size: 8px;
    }
    .kalender-info-btn {
        background: none;
        border: none;
        padding: 0;
        margin-right:30px;
        cursor: pointer;
        vertical-align: middle;
    }
    .kalender-info-btn img {
        width: 12px;
        height: 12px;
        display: inline-block;
    }
    @media (max-width: 300px) {
        .kalender-canvas { width: 98vw; height: 260px; }
    }
    </style>
</head>
<body oncontextmenu="return false">
<div class="kalender-canvas">
    <div class="kalender-header"><?= htmlspecialchars($wochentag) ?></div>
    <div class="kalender-body">
        <div class="kalender-zeile kalender-info">
            KW <?= $kalenderwoche ?> | Tag <?= $tag_im_jahr ?> | <?= $mond_emoji ?> <?= $mondphase_percent ?>% <?= $mond_text ?>
        </div>
        <div class="kalender-zeile kalender-mondphase-next">
            <?= $mondphase_next_text ?>
        </div>
        <div class="kalender-zeile kalender-sonne">
            SA: <?= $sonnenaufgang ?> / SU: <?= $sonnenuntergang ?>
        </div>
        <div class="kalender-zeile kalender-jahreszeit">
            <?= htmlspecialchars($jahreszeit) ?>
        </div>
        <div class="kalender-zeile kalender-tag">
            <?= $tag ?>
        </div>
        <div class="kalender-zeile kalender-feiertag">
            <?= $feiertag_ausgabe ?>
        </div>
        <div class="kalender-zeile kalender-monatjahr">
            <?= $monat_name . " " . $jahr ?>
        </div>
        <div class="kalender-zeile kalender-sternzeichen">
            Sternz.: <?= $sternzeichen ?>
        </div>
        <div class="kalender-footer">
            <span>
                &copy; S.G.Schiffer 2025
                <a href="https://www.sgs-visual.de/einfaches_kalenderblatt.php" target="_blank">sgs-visual.de</a>
            </span>
            <button class="kalender-info-btn" id="hilfeButton" title="Hilfe">
                <img src="kalenderdaten/info_button.png" alt="Information">
            </button>
        </div>
    </div>
</div>
<!-- Info-Dialog -->
<script>
// --- Showbox für die Abkürzungen mit Vanilla JS  ---
function showExplanationDialog() {
    // Prüfen, ob eine Dialogbox bereits existiert. Wenn ja, entfernen und Funktion beenden.
    var existingBox = document.querySelector('.custom-dialog-box');
    if (existingBox) {
        existingBox.remove();
        return;
    }

    var dialogHtml = `
        <div class="custom-dialog-box" role="dialog" aria-labelledby="customDialogHeader"
            style="position: absolute; background: #fff; border: 1px solid #555; border-radius: 4px; width: 200px; box-shadow: 0 4px 12px rgba(0,0,0,0.2); z-index: 9999; display: none; font-family: Arial, sans-serif;">
            <div class="custom-dialog-header" id="customDialogHeader" style="background:#2C79DC;color:#fff;font-size:0.8rem;font-weight:600;padding:6px 10px;border-radius:3px 3px 0 0;">
                Abkürzungen
            </div>
            <div class="custom-dialog-content" style="background:#fff;color:#000;font-size:0.75rem;padding:10px;word-break:break-word;">
                <b>KW:</b> Kalenderwoche<br>
                <b>MP:</b> Aktuelle Mondphase in Prozent<br>
                <b>abn. Mond:</b> abnehmender Mond<br>
                <b>zun. Mond:</b> zunehmender Mond<br>
                <b>SA:</b> Sonnenaufgang<br>
                <b>SU:</b> Sonnenuntergang<br>
                <b>Sternz.:</b> Aktuelles Sternzeichen<br><br>
                <sub style="font-size:0.65rem;">&copy; S.G.Schiffer - <a class="Link" href="https://www.sgs-visual.de/einfaches_kalenderblatt.php" target="_blank" style="color:#FF0000;text-decoration:none;">sgs-visual.de</a></sub>
            </div>
            <div class="custom-dialog-footer" style="text-align:right;background:#f8f8f8;border-radius:0 0 3px 3px;padding:6px 10px; border-top: 1px solid #ddd;">
                <button class="custom-dialog-ok-btn" type="button" style="background:#000;color:#fff;font-size:0.8rem;font-weight:600;border:none;border-radius:3px;padding:4px 12px;cursor:pointer;">OK</button>
            </div>
        </div>
    `;
    var temp = document.createElement('div');
    temp.innerHTML = dialogHtml;
    var box = temp.firstElementChild;
    document.body.appendChild(box);

    // Dialog positionieren
    var calendar = document.querySelector('.kalender-canvas');
    var calendarRect = calendar.getBoundingClientRect();

    box.style.display = 'block'; // Sichtbar machen, um die Dimensionen zu messen
    var boxRect = box.getBoundingClientRect();

    // Position berechnen (relativ zum Viewport, dann Scroll-Offset hinzufügen)
    var topPos = window.scrollY + calendarRect.top - boxRect.height - 5; // 5px Abstand oben
    var leftPos = window.scrollX + calendarRect.left + (calendarRect.width / 2) - (boxRect.width / 2);

    // Sicherstellen, dass die Box nicht außerhalb des sichtbaren Bereichs ist
    if (topPos < window.scrollY) { topPos = window.scrollY + 5; }
    if (leftPos < window.scrollX) { leftPos = window.scrollX + 5; }
    if (leftPos + boxRect.width > window.scrollX + document.documentElement.clientWidth) {
        leftPos = window.scrollX + document.documentElement.clientWidth - boxRect.width - 5;
    }

    box.style.top = topPos + 'px';
    box.style.left = leftPos + 'px';

    // Fade in
    box.style.opacity = 0;
    setTimeout(function() {
        box.style.transition = 'opacity 0.12s';
        box.style.opacity = 1;
    }, 10);

    // Schließen bei Klick auf OK
    function closeDialog() {
        box.style.opacity = 0;
        setTimeout(function() {
            box.remove();
        }, 120);
    }
    box.querySelector('.custom-dialog-ok-btn').addEventListener('click', closeDialog);

    // Fokus auf OK-Button
    box.querySelector('.custom-dialog-ok-btn').focus();
}

document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('hilfeButton').addEventListener('click', showExplanationDialog);
});
</script>
<!--  Geschafft - Script endlich vollständig  -->
</body>
</html>